"""Accident APIs."""

from typing import List

from fastapi import (
    APIRouter,
    HTTPException,
    Request,
)

from app.api.v1.auth import get_current_user
from app.core.config import settings
from app.core.limiter import limiter
from app.core.logging import logger
from app.schemas.accident import AccidentResponse
from app.services.database import DatabaseService
from app.utils.sanitization import sanitize_string

router = APIRouter()
db_service = DatabaseService()


@router.get("/all", response_model=List[AccidentResponse])
# @limiter.limit(settings.RATE_LIMIT_ENDPOINTS["messages"][0])
async def get_maps(
    request: Request,
):
    """Get all accidents data.

    Args:
        request: The FastAPI request object for rate limiting.

    Returns:
        List[AccidentResponse]: List of accident info.

    Raises:
        HTTPException: If there's an error retrieving the accidents.
    """
    try:
        accidents = await db_service.get_accident()
        return [
            AccidentResponse(
                id=sanitize_string(accident.id),
                year=sanitize_string(accident.year),
                all_accident=sanitize_string(accident.all_accident),
                leading_to_death=sanitize_string(accident.leading_to_death),
                lead_to_injury=sanitize_string(accident.lead_to_injury),
                lead_to_damage=sanitize_string(accident.lead_to_damage),
            )
            for accident in accidents
        ]
    except ValueError as ve:
        logger.error("get_accident_validation_failed", error=str(ve), exc_info=True)
        raise HTTPException(status_code=422, detail=str(ve))
